<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Http\Requests\EnrollMatrixRequest;
use App\Enums\Matrix\PinStatus;
use App\Enums\Transaction\Source;
use App\Enums\Transaction\Type;
use App\Enums\Transaction\WalletType;
use App\Enums\Email\EmailSmsTemplateName;
use App\Enums\Payment\NotificationType;
use App\Services\Investment\MatrixInvestmentService;
use App\Services\Investment\MatrixService;
use App\Services\Payment\WalletService;
use App\Services\Payment\TransactionService;
use App\Services\Payment\DepositService;
use App\Services\UserService;
use App\Services\PinGenerateService;
use App\Services\SettingService;
use App\Jobs\EmailSmsProcessJob;
use App\Notifications\DepositNotification;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;
use Illuminate\Support\Facades\DB;

class MatrixController extends Controller
{
    public function __construct(
        protected MatrixService  $matrixService,
        protected WalletService  $walletService,
        protected MatrixInvestmentService $matrixInvestmentService,
        protected UserService $userService,
        protected DepositService $depositService,
        protected PinGenerateService $pinGenerateService,
        protected TransactionService $transactionService,
    ){

    }

    public function index(): View
    {
        $setTitle = "Matrix";

        $userId = (int)Auth::id();
        $matrixLog = $this->matrixInvestmentService->findByUserId($userId);

        return view('user.matrix.index', compact(
            'setTitle',
            'matrixLog'
        ));
    }

    /**
     * @param EnrollMatrixRequest $request
     * @return RedirectResponse
     */
    public function store(EnrollMatrixRequest $request): RedirectResponse
    {
		$setting = SettingService::getSetting();

        if (getArrayValue($setting->system_configuration, 'e_pin.value') != \App\Enums\Status::ACTIVE->value){
            abort(404);
        }
		
        try{
			$setting = SettingService::getSetting();
            $pinNumber = $this->pinGenerateService->findByPinNumber($request->input('pin_number'));
            
            if (!$pinNumber) {
                return back()->with('notify', [['error', 'Subscription Code not found']]);
            }

            if ($pinNumber->status == PinStatus::USED->value) {
                return back()->with('notify', [['error', 'Subscription Code already used.']]);
            }
			
            $user = $this->userService->findById((int)Auth::id());
            $wallet = $user->wallet;
			$plan = $this->matrixService->findByUid($request->input('uid'));
			if($pinNumber->amount != $plan->amount){
				return back()->with('notify', [['error', 'Not a valid subscription Code.']]);
			}
			DB::transaction(function () use ($pinNumber, $wallet, $user, $plan, $request) {

                $amount = $pinNumber->amount;
                $half_amt = $pinNumber->amount/2;

                $pinNumber->status = PinStatus::USED->value;
                $pinNumber->user_id = Auth::id();
                $pinNumber->save();

                $wallet->primary_balance += $half_amt;
                $wallet->trade_balance += $half_amt;
                $wallet->save();

                $this->transactionService->save(
                    $this->transactionService->prepParams([
                        'user_id' => Auth::id(),
                        'amount' => $half_amt,
                        'type' => Type::PLUS,
                        'details' => 'Top up Code with amount ' . getCurrencySymbol() . shortAmount($half_amt),
                        'wallet' => $this->walletService->findBalanceByWalletType(WalletType::PRIMARY->value, $wallet),
                        'source' => Source::ALL->value
                    ])
                );
                $this->transactionService->save(
                    $this->transactionService->prepParams([
                        'user_id' => Auth::id(),
                        'amount' => $half_amt,
                        'type' => Type::PLUS,
                        'details' => 'Top up Code with amount ' . getCurrencySymbol() . shortAmount($half_amt),
                        'wallet' => $this->walletService->findBalanceByWalletType(WalletType::TRADE->value, $wallet),
                        'source' => Source::ALL->value
                    ])
                );

                dispatch(new EmailSmsProcessJob($user, [
                    'amount' => $amount,
                    'pin_number' => $pinNumber->pin_number,
                    'currency' => getCurrencySymbol(),
                ], EmailSmsTemplateName::PIN_RECHARGE->value));


                $deposit = $this->depositService->save($this->depositService->depositPrepParams($amount));
                $deposit->notify(new DepositNotification(NotificationType::APPROVED));
			
				$this->matrixInvestmentService->executeEnrolledScheme($request, $plan, $user, $request->input('pin_number'));
            });
        }catch (\Exception $exception){
            return back()->with('notify', [['error', $exception->getMessage()]]);
        }

        return back()->with('notify', [['success', "Successfully you have been enrolled."]]);
    }
}
