<?php

namespace App\Services\Investment;

use App\Actions\MatrixHandler;
use App\Enums\Matrix\InvestmentStatus;
use App\Enums\Transaction\Source;
use App\Enums\Transaction\Type;
use App\Enums\Transaction\WalletType;
use App\Models\MatrixInvestment;
use App\Models\MatrixInvestmentDetails;
use App\Models\Matrix;
use App\Models\User;
use App\Notifications\MatrixInvestmentNotification;
use App\Services\Payment\TransactionService;
use App\Services\Payment\WalletService;
use App\Services\PinGenerateService;
use App\Services\SettingService;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\Request;
use Illuminate\Pagination\AbstractPaginator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class MatrixInvestmentService
{
    public function __construct(
        protected TransactionService $transactionService,
        protected WalletService $walletService,
        protected PinGenerateService $pinGenerateService,
    )
    {

    }

    /**
     * @param Matrix $matrix
     * @param int $userId
     * @return array
     */
    public function prepParams(Matrix $matrix, int $userId, $pin_number): array
    {
        $level = [
            'matrix_levels' => array_map(fn($item) => [
                'level' => $item['level'],
                'amount' => getAmount($item['amount']),
            ], $matrix->matrixLevel->toArray()),
        ];

        return [
            'uid' => Str::random(),
            'user_id' => $userId,
            'plan_id' => $matrix->id,
            'name' => $matrix->name,
            'trx' => getTrx(),
            'price' => $matrix->amount,
            'referral_reward' => $matrix->referral_reward,
            'meta' => $level,
            'pin_number' => $pin_number,
            'status' => InvestmentStatus::RUNNING->value,
        ];
    }
    
public function invest_details_save(MatrixInvestment $invest, Matrix $matrix, $user, $pin_number): MatrixInvestmentDetails
{
    $setting = SettingService::getSetting();
    $franchaisee_commission_amount = getArrayValue($setting->commissions_charge, 'franchisee_commissions');
    $pinNumber = $this->pinGenerateService->findByPinNumber($pin_number);
    
    $data = [
        'matrix_investments_id' => $invest->id,
        'franchaisee_id' => $user?->franchisee_id,
        'pin_amount' => $pinNumber->amount,
        'franchaisee_commission_amount' => $pinNumber->amount * ($franchaisee_commission_amount / 100),
    ];

    $matrix_invest_details = MatrixInvestmentDetails::create($data);

    if ($user->franchisee_id != null) {
        $franchisee_user = User::where('id', $user->franchisee_id)->first();
        $investing_user = User::find($invest->user_id); // Assuming $invest is the MatrixInvestment model instance

        if ($investing_user) {
            $wallet = $franchisee_user->wallet;
            $wallet->primary_balance += $matrix_invest_details->franchaisee_commission_amount;
            $wallet->save();

            // Transaction creation
            $this->transactionService->save([
                'user_id' => $franchisee_user->id,
                'amount' => $matrix_invest_details->franchaisee_commission_amount,
                'post_balance' => $wallet->primary_balance,
                'charge' => 0,
                'trx' => getTrx(),
                'type' => Type::PLUS->value,
                'wallet_type' => WalletType::PRIMARY->value,
                'source' => Source::REFERRAL->value,
                'details' => 'Franchisee commission received from ' . $investing_user->username,
            ]);

            return $matrix_invest_details;
        }
    }

    return $matrix_invest_details;
}


    /**
     * @param array $params
     * @return MatrixInvestment
     */
    public function save(array $params): MatrixInvestment
    {
        return MatrixInvestment::create($params);
    }

   /**
 * @param Request $request
 * @param Matrix $matrix
 * @param User $user
 * @param string $pin_number
 * @return void
 */
public function executeEnrolledScheme(Request $request, Matrix $matrix, User $user, string $pin_number): void
{
    DB::transaction(function () use ($request, $matrix, $user, $pin_number) {
        // Check if user is already enrolled
        $existingInvestment = $this->findByUserId($user->id);
        if ($existingInvestment) {
            throw new \Exception('You are already Subscribed.');
        }

        $matrixHandler = new MatrixHandler($user, $matrix);
        $matrixHandler->store();

        $invest = $this->save($this->prepParams($matrix, $user->id, $pin_number));
        $investDetails = $this->invest_details_save($invest, $matrix, $user, $pin_number);
        $invest->notify(new MatrixInvestmentNotification());

        // Deduct amount from user's wallet and log the transaction (if required)
        /*
        $wallet = $user->wallet;
        $wallet->primary_balance -= $matrix->amount;
        $wallet->save();

        $this->transactionService->save($this->transactionService->prepParams([
            'user_id' => $user->id,
            'amount' => $matrix->amount,
            'type' => Type::MINUS,
            'trx' => getTrx(),
            'details' => "Enrollment in the {$matrix->name} Matrix plan.",
            'wallet' => $this->walletService->findBalanceByWalletType(WalletType::PRIMARY->value, $wallet),
            'source' => Source::INVESTMENT->value
        ]));
        */
    });
}


    /**
     * @param int $userId
     * @return MatrixInvestment|null
     */
    public function findByUserId(int $userId): ?MatrixInvestment
    {
        return MatrixInvestment::where('user_id', $userId)->first();
    }

    /**
     * @return AbstractPaginator
     */
    public function getEnrolledByPaginate(): AbstractPaginator
    {
        return MatrixInvestment::filter(request()->all())
            ->with(['user'])
            ->paginate(getPaginate());
    }


    public function latestMatrix(array $with = [], int $limit = 5): Collection
    {
        return  MatrixInvestment::query()
            ->with($with)
            ->latest('id')
            ->take($limit)->get();
    }

    public function getMatrixReport(int|string $userId = null)
    {
        $now = now()->toDateString();
        $query = MatrixInvestment::query();

        if (!is_null($userId)) {
            $query->where('user_id', $userId);
        }

        return $query->selectRaw('
        COALESCE(SUM(price), 0) as total,
        COALESCE(SUM(CASE WHEN DATE(created_at) = ? THEN price ELSE 0 END), 0) as today,
        COALESCE(SUM(referral_commissions), 0) as referral,
        COALESCE(SUM(level_commissions), 0) as level,
        COALESCE(SUM(level_commissions + referral_commissions), 0) as commission
    ', [$now])->first();
    }

    /**
     * @return array
     */
    public function monthlyReport(): array
    {
        $report = [
            'months' => collect(),
            'invest_month_amount' => collect(),
        ];

        $startOfLast12Months = Carbon::now()->subMonths(11)->startOfMonth();

        $logs = MatrixInvestment::where('created_at', '>=', $startOfLast12Months)
            ->selectRaw("DATE_FORMAT(created_at, '%M-%Y') as months, SUM(price) as invest_amount")
            ->orderBy('months')
            ->groupBy('months')
            ->get();

        $last12Months = collect(CarbonPeriod::create($startOfLast12Months, '1 month', Carbon::now()->endOfMonth()))
            ->map(function ($date) {
                return $date->format('F-Y');
            });

        $last12Months->each(function ($month) use (&$report, $logs) {
            $logDataForMonth = $logs->firstWhere('months', $month);

            $report['months']->push($month);
            $report['invest_month_amount']->push(optional($logDataForMonth)->invest_amount ?? 0);
        });

        return [
            $report['months']->values()->all(),
            $report['invest_month_amount']->values()->all(),
        ];

    }



}


