<?php

namespace App\Http\Controllers\User;

use App\Enums\Trade\TradeType;
use App\Enums\Transaction\Type;
use App\Enums\Transaction\WalletType;
use App\Events\CryptoPriceUpdated;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Requests\TradeRequest;
use App\Services\Payment\WalletService;
use App\Services\SettingService;
use App\Services\Trade\ActivityLogService;
use App\Services\Trade\CryptoCurrencyService;
use App\Services\Trade\ParameterService;
use App\Services\UserService;
use App\Models\CompetitionParticipants;
use App\Models\Competition;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;
use Carbon\Carbon;
use App\Models\TradeLog; // Ensure this line is added to import the TradeLog model
use Illuminate\Http\RedirectResponse;

class TradeController extends Controller
{
    public function __construct(
        protected WalletService $walletService,
        protected UserService $userService,
        protected ActivityLogService $activityLogService,
        protected CryptoCurrencyService $cryptoCurrencyService,
        protected ParameterService $parameterService,
    ){
    }

public function index(): View
{
    $setTitle = "Trades";
    $cryptoCurrency = $this->cryptoCurrencyService->getActiveCryptoCurrencyByPaginate();
    
    // Get trade logs for today and group by cryptocurrency
    $todayTradeLogs = TradeLog::whereDate('created_at', today())
        ->with('cryptoCurrency') // Ensure related cryptoCurrency is loaded
        ->get()
        ->groupBy('cryptoCurrency.id'); // Group by cryptoCurrency ID

    // Prepare data to be used in the view
    $cryptoData = $todayTradeLogs->map(function ($tradeLogs, $cryptoId) {
        $totalTrades = $tradeLogs->count();
        $totalAmount = $tradeLogs->sum('amount');
        
        // Initialize virtual purchase, profit, and loss
        $totalVirtualPurchase = 0;
        $totalVirtualProfit = 0;
        $totalVirtualLoss = 0;

        // Calculate virtual purchase, profit, and loss
        $tradeLogs->each(function ($tradeLog) use (&$totalVirtualPurchase, &$totalVirtualProfit, &$totalVirtualLoss) {
            $priceWas = $tradeLog->original_price;
            $resultPrice = $tradeLog->meta['result_price'] ?? $tradeLog->result_price;
            $difference = $resultPrice - $priceWas;
            $virtualPurchase = $tradeLog->amount / $priceWas;

            $totalVirtualPurchase += $virtualPurchase;

            if ($tradeLog->outcome == \App\Enums\Trade\TradeOutcome::WIN->value) {
                $totalVirtualProfit += ($difference * $virtualPurchase);
            } elseif ($tradeLog->outcome == \App\Enums\Trade\TradeOutcome::LOSE->value) {
                $totalVirtualLoss += ($difference * $virtualPurchase);
            }
        });

        $virtualProfitLoss = $totalVirtualProfit - $totalVirtualLoss;

        // Prepare the detailed trade logs for each crypto
        $detailedTradeLogs = $tradeLogs->map(function ($tradeLog) {
            $priceWas = $tradeLog->original_price;
            $resultPrice = $tradeLog->meta['result_price'] ?? $tradeLog->result_price;
            $difference = $resultPrice - $priceWas;
            $differenceFormatted = number_format(abs($difference), 6);
            $isProfit = $tradeLog->outcome == \App\Enums\Trade\TradeOutcome::WIN->value;
            $isDraw = $tradeLog->outcome == \App\Enums\Trade\TradeOutcome::DRAW->value;
            $isInitiated = $tradeLog->outcome == \App\Enums\Trade\TradeOutcome::INITIATED->value;
            $purchasedVolume = $priceWas ? $tradeLog->amount / $priceWas : 0;
            $purchasedVolumeFormatted = number_format($purchasedVolume, 6);

            return [
                'initiatedAt' => $tradeLog->created_at->diffForHumans(),
                'amount' => $tradeLog->amount,
                'volume' => $tradeLog->volume,
                'priceWas' => $priceWas,
                'purchasedVolume' => $purchasedVolumeFormatted,
                'resultPrice' => $resultPrice,
                'difference' => $isInitiated ? 'N/A' : ($isProfit ? '+' : ($isDraw ? '' : '-')) . getCurrencySymbol() . $differenceFormatted,
                'outcome' => \App\Enums\Trade\TradeOutcome::getName($tradeLog->outcome),
                'outcomeClass' => \App\Enums\Trade\TradeOutcome::getColor($tradeLog->outcome)
            ];
        });

        return [
            'crypto' => $tradeLogs->first()->cryptoCurrency,
            'totalTrades' => $totalTrades,
            'totalAmount' => $totalAmount,
            'totalVirtualPurchase' => $totalVirtualPurchase,
            'totalVirtualProfit' => $totalVirtualProfit,
            'totalVirtualLoss' => $totalVirtualLoss,
            'virtualProfitLoss' => $virtualProfitLoss,
            'detailedTradeLogs' => $detailedTradeLogs
        ];
    });

    return view('user.trade.index', compact(
        'setTitle',
        'cryptoCurrency',
        'cryptoData'
    ));
}




    public function tradeLog(): View
    {
        $setTitle = "Trade logs";

        $userId = (int)Auth::id();
        [$days, $amount] = $this->activityLogService->dayReport($userId);
        $trade = $this->activityLogService->getTradeReport($userId);
        $tradeLogs = $this->activityLogService->getByUser($userId, TradeType::TRADE);

        return view('user.trade.trade_log', compact(
            'setTitle',
            'tradeLogs',
            'trade',
            'days',
            'amount',
        ));
    }

    public function practiceLog(): View
    {
        $setTitle = "Practice logs";
        $userId = Auth::id();
        $practiceLogs = $this->activityLogService->getByUser($userId, TradeType::PRACTICE);

        return view('user.trade.practice_log', compact(
            'setTitle',
            'practiceLogs',
        ));
    }

   /**
     * @param string $pair
     * @return View
     */
     
     
/*

public function trade(string $pair): View|RedirectResponse
{
    // Fetch the setting to determine if binary trade commissions should be applied
    $settings = SettingService::getSetting();
    $applyBinaryTradeCommissions = $settings->binary_trade_set;

    if (getArrayValue($settings->system_configuration, 'binary_trade.value') != \App\Enums\Status::ACTIVE->value) {
        abort(404);
    }

    $setTitle = "Trade now";

    $userId = (int)Auth::id();
    $crypto = $this->cryptoCurrencyService->findByPair($pair);
    $parameters = $this->parameterService->activeParameter();

    // Check if the user has already made a trade today if binary_trade_set is enabled
    if ($applyBinaryTradeCommissions) {
        $today = now()->startOfDay();
        $hasTradedToday = TradeLog::where('user_id', $userId)
                            ->where('type', TradeType::TRADE->value)
                            ->where('created_at', '>=', $today)
                            ->exists();

        if ($hasTradedToday) {
            // Redirect to the trading page with an error message
            $notify[] = ['error', 'You can only trade once per day.'];
            return redirect()->route('user.trade.index')->withNotify($notify);
        }
    }

    $tradeLogs = $this->activityLogService->getByUser($userId, TradeType::TRADE, true);

    // Add a success notification when the user can proceed to the trading page
    $notify[] = ['success', "Trade has been generated"];
    return view('user.trade.trading', compact(
        'setTitle',
        'crypto',
        'parameters',
        'tradeLogs',
    ))->withNotify($notify);
}
*/
     
     
 public function trade(string $pair): View
{
    $setting = SettingService::getSetting();
    if (getArrayValue($setting->system_configuration, 'binary_trade.value') != \App\Enums\Status::ACTIVE->value) {
        abort(404);
    }

    $setTitle = "Trade now";

    $userId = (int)Auth::id();
    $user = Auth::user(); // Get the authenticated user
    $crypto = $this->cryptoCurrencyService->findByPair($pair);
    $parameters = $this->parameterService->activeParameter();
    $tradeLogs = $this->activityLogService->getByUser($userId, TradeType::TRADE, true);

    // Fetch the trade balance
    $tradeBalance = $user->wallet->trade_balance;

    return view('user.trade.trading', compact(
        'setTitle',
        'crypto',
        'parameters',
        'tradeLogs',
        'tradeBalance' // Pass trade balance to the view
    ));
}


    /**
     * @param string $pair
     * @return View
     */
    public function practice(string $pair): View
    {
        $setting = SettingService::getSetting();

        if (getArrayValue($setting->system_configuration, 'practice_trade.value') != \App\Enums\Status::ACTIVE->value) {
            abort(404);
        }

        $setTitle = "Practice now";
        $userId = (int)Auth::id();
        $crypto = $this->cryptoCurrencyService->findByPair($pair);
        $parameters = $this->parameterService->activeParameter();
        $tradeLogs = $this->activityLogService->getByUser($userId, TradeType::PRACTICE, true);

        return view('user.trade.trading', compact(
            'setTitle',
            'crypto',
            'parameters',
            'tradeLogs',
        ));
    }

    /**
     * @throws Exception
     */
    public function store(TradeRequest $request, $id)
    {
        try {
            $parameter = $this->parameterService->findById($request->integer('parameter_id'));
            $crypto = $this->cryptoCurrencyService->findById((int)$id);

            if(!$parameter || !$crypto){
                abort(404);
            }
			
			if($request->integer('type') == TradeType::VIRTUAL_TRADE->value){
				$checkCompetitionTime = Competition::where('id', $request->input('competition_id'))->first();
				$currentDate = Carbon::now();
				$startDate = Carbon::parse($checkCompetitionTime->start_time);
				$endDate = Carbon::parse($checkCompetitionTime->end_time);
				if(!$currentDate->between($startDate, $endDate)){
					throw new Exception("Competition time is over.");
				}
				
				$chkCompetitionBalance = $this->walletService->chkCompetitionBalance($request->input('amount'),$request->input('competition_id'));
			}
			
            $walletType = $request->integer('type') == TradeType::TRADE->value ? WalletType::TRADE->value : ($request->integer('type') == TradeType::VIRTUAL_TRADE->value ? WalletType::VIRTUAL_TRADE->value : WalletType::PRACTICE->value);

            [$wallet, $account] = $this->walletService->checkWalletBalance($request->input('amount'), $walletType, true);
            $this->activityLogService->executeTrade($request, $wallet, $account, Type::MINUS, $parameter, $crypto);
			
			if($request->integer('type') == TradeType::VIRTUAL_TRADE->value){
				$user = (int)Auth::id();
				$balance = CompetitionParticipants::where('user_id', $user)->where('competition_id', $request->input('competition_id'))->first();
				CompetitionParticipants::where('user_id', $user)->where('competition_id', $request->input('competition_id'))->update([
					'competition_trade_amt'=>floatval($balance->competition_trade_amt) - floatval($request->input('amount'))
				]);
			}
			
            $notify[] = ['success', "Trade has been generated"];
            return back()->withNotify($notify);

        }catch (Exception $exception){

            $notify[] = ['warning', $exception->getMessage()];
            return back()->withNotify($notify);
        }
    }
    public function trade_details(Request $request)
    {
		$pair = $request->crypto_id;
        $userId = (int)Auth::id();
        $data['crypto'] = $this->cryptoCurrencyService->findByPair($pair);
        $data['parameters'] = $this->parameterService->activeParameter();
        $data['tradeLogs'] = $this->activityLogService->getByUser($userId, TradeType::PRACTICE, true);
		return response(['status' => 200, 'data' => $data]);
    }
}
